/*
 * Copyright (C) 2020 Pascal Knecht
 * Copyright (C) 2020 Méline Sieber
 *
 * Copyright (C) secunet Security Networks AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include <test_suite.h>

#include "tls_hkdf.h"

static chunk_t ulfheim_ecdhe = chunk_from_chars(
	0xdf,0x4a,0x29,0x1b,0xaa,0x1e,0xb7,0xcf,0xa6,0x93,0x4b,0x29,0xb4,0x74,0xba,0xad,
	0x26,0x97,0xe2,0x9f,0x1f,0x92,0x0d,0xcc,0x77,0xc8,0xa0,0xa0,0x88,0x44,0x76,0x24,
);

static chunk_t ulfheim_client_server_hello = chunk_from_chars(
	/* Client Hello */
	0x01,0x00,0x00,0xc6,0x03,0x03,0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
	0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,
	0x1a,0x1b,0x1c,0x1d,0x1e,0x1f,0x20,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
	0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,
	0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff,0x00,0x06,0x13,0x01,0x13,0x02,0x13,0x03,0x01,
	0x00,0x00,0x77,0x00,0x00,0x00,0x18,0x00,0x16,0x00,0x00,0x13,0x65,0x78,0x61,0x6d,
	0x70,0x6c,0x65,0x2e,0x75,0x6c,0x66,0x68,0x65,0x69,0x6d,0x2e,0x6e,0x65,0x74,0x00,
	0x0a,0x00,0x08,0x00,0x06,0x00,0x1d,0x00,0x17,0x00,0x18,0x00,0x0d,0x00,0x14,0x00,
	0x12,0x04,0x03,0x08,0x04,0x04,0x01,0x05,0x03,0x08,0x05,0x05,0x01,0x08,0x06,0x06,
	0x01,0x02,0x01,0x00,0x33,0x00,0x26,0x00,0x24,0x00,0x1d,0x00,0x20,0x35,0x80,0x72,
	0xd6,0x36,0x58,0x80,0xd1,0xae,0xea,0x32,0x9a,0xdf,0x91,0x21,0x38,0x38,0x51,0xed,
	0x21,0xa2,0x8e,0x3b,0x75,0xe9,0x65,0xd0,0xd2,0xcd,0x16,0x62,0x54,0x00,0x2d,0x00,
	0x02,0x01,0x01,0x00,0x2b,0x00,0x03,0x02,0x03,0x04,
	/* Server Hello */
	0x02,0x00,0x00,0x76,0x03,0x03,0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,
	0x7a,0x7b,0x7c,0x7d,0x7e,0x7f,0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,
	0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,0x20,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,
	0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,
	0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff,0x13,0x01,0x00,0x00,0x2e,0x00,0x33,0x00,0x24,
	0x00,0x1d,0x00,0x20,0x9f,0xd7,0xad,0x6d,0xcf,0xf4,0x29,0x8d,0xd3,0xf9,0x6d,0x5b,
	0x1b,0x2a,0xf9,0x10,0xa0,0x53,0x5b,0x14,0x88,0xd7,0xf8,0xfa,0xbb,0x34,0x9a,0x98,
	0x28,0x80,0xb6,0x15,0x00,0x2b,0x00,0x02,0x03,0x04,
);

static chunk_t ulfheim_server_data = chunk_from_chars(
	/* Server Encrypted Extension */
	0x08,0x00,0x00,0x02,0x00,0x00,
	/* Server Certificate */
	0x0b,0x00,0x03,0x2e,0x00,0x00,0x03,0x2a,0x00,0x03,0x25,0x30,0x82,0x03,0x21,0x30,
	0x82,0x02,0x09,0xa0,0x03,0x02,0x01,0x02,0x02,0x08,0x15,0x5a,0x92,0xad,0xc2,0x04,
	0x8f,0x90,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,
	0x00,0x30,0x22,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,
	0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x0a,0x13,0x0a,0x45,0x78,0x61,0x6d,0x70,
	0x6c,0x65,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x31,0x38,0x31,0x30,0x30,0x35,0x30,
	0x31,0x33,0x38,0x31,0x37,0x5a,0x17,0x0d,0x31,0x39,0x31,0x30,0x30,0x35,0x30,0x31,
	0x33,0x38,0x31,0x37,0x5a,0x30,0x2b,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,
	0x13,0x02,0x55,0x53,0x31,0x1c,0x30,0x1a,0x06,0x03,0x55,0x04,0x03,0x13,0x13,0x65,
	0x78,0x61,0x6d,0x70,0x6c,0x65,0x2e,0x75,0x6c,0x66,0x68,0x65,0x69,0x6d,0x2e,0x6e,
	0x65,0x74,0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
	0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,
	0x01,0x01,0x00,0xc4,0x80,0x36,0x06,0xba,0xe7,0x47,0x6b,0x08,0x94,0x04,0xec,0xa7,
	0xb6,0x91,0x04,0x3f,0xf7,0x92,0xbc,0x19,0xee,0xfb,0x7d,0x74,0xd7,0xa8,0x0d,0x00,
	0x1e,0x7b,0x4b,0x3a,0x4a,0xe6,0x0f,0xe8,0xc0,0x71,0xfc,0x73,0xe7,0x02,0x4c,0x0d,
	0xbc,0xf4,0xbd,0xd1,0x1d,0x39,0x6b,0xba,0x70,0x46,0x4a,0x13,0xe9,0x4a,0xf8,0x3d,
	0xf3,0xe1,0x09,0x59,0x54,0x7b,0xc9,0x55,0xfb,0x41,0x2d,0xa3,0x76,0x52,0x11,0xe1,
	0xf3,0xdc,0x77,0x6c,0xaa,0x53,0x37,0x6e,0xca,0x3a,0xec,0xbe,0xc3,0xaa,0xb7,0x3b,
	0x31,0xd5,0x6c,0xb6,0x52,0x9c,0x80,0x98,0xbc,0xc9,0xe0,0x28,0x18,0xe2,0x0b,0xf7,
	0xf8,0xa0,0x3a,0xfd,0x17,0x04,0x50,0x9e,0xce,0x79,0xbd,0x9f,0x39,0xf1,0xea,0x69,
	0xec,0x47,0x97,0x2e,0x83,0x0f,0xb5,0xca,0x95,0xde,0x95,0xa1,0xe6,0x04,0x22,0xd5,
	0xee,0xbe,0x52,0x79,0x54,0xa1,0xe7,0xbf,0x8a,0x86,0xf6,0x46,0x6d,0x0d,0x9f,0x16,
	0x95,0x1a,0x4c,0xf7,0xa0,0x46,0x92,0x59,0x5c,0x13,0x52,0xf2,0x54,0x9e,0x5a,0xfb,
	0x4e,0xbf,0xd7,0x7a,0x37,0x95,0x01,0x44,0xe4,0xc0,0x26,0x87,0x4c,0x65,0x3e,0x40,
	0x7d,0x7d,0x23,0x07,0x44,0x01,0xf4,0x84,0xff,0xd0,0x8f,0x7a,0x1f,0xa0,0x52,0x10,
	0xd1,0xf4,0xf0,0xd5,0xce,0x79,0x70,0x29,0x32,0xe2,0xca,0xbe,0x70,0x1f,0xdf,0xad,
	0x6b,0x4b,0xb7,0x11,0x01,0xf4,0x4b,0xad,0x66,0x6a,0x11,0x13,0x0f,0xe2,0xee,0x82,
	0x9e,0x4d,0x02,0x9d,0xc9,0x1c,0xdd,0x67,0x16,0xdb,0xb9,0x06,0x18,0x86,0xed,0xc1,
	0xba,0x94,0x21,0x02,0x03,0x01,0x00,0x01,0xa3,0x52,0x30,0x50,0x30,0x0e,0x06,0x03,
	0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x05,0xa0,0x30,0x1d,0x06,0x03,
	0x55,0x1d,0x25,0x04,0x16,0x30,0x14,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,
	0x02,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x03,0x01,0x30,0x1f,0x06,0x03,0x55,
	0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x89,0x4f,0xde,0x5b,0xcc,0x69,0xe2,0x52,
	0xcf,0x3e,0xa3,0x00,0xdf,0xb1,0x97,0xb8,0x1d,0xe1,0xc1,0x46,0x30,0x0d,0x06,0x09,
	0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x82,0x01,0x01,0x00,
	0x59,0x16,0x45,0xa6,0x9a,0x2e,0x37,0x79,0xe4,0xf6,0xdd,0x27,0x1a,0xba,0x1c,0x0b,
	0xfd,0x6c,0xd7,0x55,0x99,0xb5,0xe7,0xc3,0x6e,0x53,0x3e,0xff,0x36,0x59,0x08,0x43,
	0x24,0xc9,0xe7,0xa5,0x04,0x07,0x9d,0x39,0xe0,0xd4,0x29,0x87,0xff,0xe3,0xeb,0xdd,
	0x09,0xc1,0xcf,0x1d,0x91,0x44,0x55,0x87,0x0b,0x57,0x1d,0xd1,0x9b,0xdf,0x1d,0x24,
	0xf8,0xbb,0x9a,0x11,0xfe,0x80,0xfd,0x59,0x2b,0xa0,0x39,0x8c,0xde,0x11,0xe2,0x65,
	0x1e,0x61,0x8c,0xe5,0x98,0xfa,0x96,0xe5,0x37,0x2e,0xef,0x3d,0x24,0x8a,0xfd,0xe1,
	0x74,0x63,0xeb,0xbf,0xab,0xb8,0xe4,0xd1,0xab,0x50,0x2a,0x54,0xec,0x00,0x64,0xe9,
	0x2f,0x78,0x19,0x66,0x0d,0x3f,0x27,0xcf,0x20,0x9e,0x66,0x7f,0xce,0x5a,0xe2,0xe4,
	0xac,0x99,0xc7,0xc9,0x38,0x18,0xf8,0xb2,0x51,0x07,0x22,0xdf,0xed,0x97,0xf3,0x2e,
	0x3e,0x93,0x49,0xd4,0xc6,0x6c,0x9e,0xa6,0x39,0x6d,0x74,0x44,0x62,0xa0,0x6b,0x42,
	0xc6,0xd5,0xba,0x68,0x8e,0xac,0x3a,0x01,0x7b,0xdd,0xfc,0x8e,0x2c,0xfc,0xad,0x27,
	0xcb,0x69,0xd3,0xcc,0xdc,0xa2,0x80,0x41,0x44,0x65,0xd3,0xae,0x34,0x8c,0xe0,0xf3,
	0x4a,0xb2,0xfb,0x9c,0x61,0x83,0x71,0x31,0x2b,0x19,0x10,0x41,0x64,0x1c,0x23,0x7f,
	0x11,0xa5,0xd6,0x5c,0x84,0x4f,0x04,0x04,0x84,0x99,0x38,0x71,0x2b,0x95,0x9e,0xd6,
	0x85,0xbc,0x5c,0x5d,0xd6,0x45,0xed,0x19,0x90,0x94,0x73,0x40,0x29,0x26,0xdc,0xb4,
	0x0e,0x34,0x69,0xa1,0x59,0x41,0xe8,0xe2,0xcc,0xa8,0x4b,0xb6,0x08,0x46,0x36,0xa0,
	0x00,0x00,
	/* Server Certificate Verify */
	0x0f,0x00,0x01,0x04,0x08,0x04,0x01,0x00,0x17,0xfe,0xb5,0x33,0xca,0x6d,0x00,0x7d,
	0x00,0x58,0x25,0x79,0x68,0x42,0x4b,0xbc,0x3a,0xa6,0x90,0x9e,0x9d,0x49,0x55,0x75,
	0x76,0xa5,0x20,0xe0,0x4a,0x5e,0xf0,0x5f,0x0e,0x86,0xd2,0x4f,0xf4,0x3f,0x8e,0xb8,
	0x61,0xee,0xf5,0x95,0x22,0x8d,0x70,0x32,0xaa,0x36,0x0f,0x71,0x4e,0x66,0x74,0x13,
	0x92,0x6e,0xf4,0xf8,0xb5,0x80,0x3b,0x69,0xe3,0x55,0x19,0xe3,0xb2,0x3f,0x43,0x73,
	0xdf,0xac,0x67,0x87,0x06,0x6d,0xcb,0x47,0x56,0xb5,0x45,0x60,0xe0,0x88,0x6e,0x9b,
	0x96,0x2c,0x4a,0xd2,0x8d,0xab,0x26,0xba,0xd1,0xab,0xc2,0x59,0x16,0xb0,0x9a,0xf2,
	0x86,0x53,0x7f,0x68,0x4f,0x80,0x8a,0xef,0xee,0x73,0x04,0x6c,0xb7,0xdf,0x0a,0x84,
	0xfb,0xb5,0x96,0x7a,0xca,0x13,0x1f,0x4b,0x1c,0xf3,0x89,0x79,0x94,0x03,0xa3,0x0c,
	0x02,0xd2,0x9c,0xbd,0xad,0xb7,0x25,0x12,0xdb,0x9c,0xec,0x2e,0x5e,0x1d,0x00,0xe5,
	0x0c,0xaf,0xcf,0x6f,0x21,0x09,0x1e,0xbc,0x4f,0x25,0x3c,0x5e,0xab,0x01,0xa6,0x79,
	0xba,0xea,0xbe,0xed,0xb9,0xc9,0x61,0x8f,0x66,0x00,0x6b,0x82,0x44,0xd6,0x62,0x2a,
	0xaa,0x56,0x88,0x7c,0xcf,0xc6,0x6a,0x0f,0x38,0x51,0xdf,0xa1,0x3a,0x78,0xcf,0xf7,
	0x99,0x1e,0x03,0xcb,0x2c,0x3a,0x0e,0xd8,0x7d,0x73,0x67,0x36,0x2e,0xb7,0x80,0x5b,
	0x00,0xb2,0x52,0x4f,0xf2,0x98,0xa4,0xda,0x48,0x7c,0xac,0xde,0xaf,0x8a,0x23,0x36,
	0xc5,0x63,0x1b,0x3e,0xfa,0x93,0x5b,0xb4,0x11,0xe7,0x53,0xca,0x13,0xb0,0x15,0xfe,
	0xc7,0xe4,0xa7,0x30,0xf1,0x36,0x9f,0x9e,
	/* Server Handshake Finish */
	0x14,0x00,0x00,0x20,0xea,0x6e,0xe1,0x76,0xdc,0xcc,0x4a,0xf1,0x85,0x9e,0x9e,0x4e,
	0x93,0xf7,0x97,0xea,0xc9,0xa7,0x8c,0xe4,0x39,0x30,0x1e,0x35,0x27,0x5a,0xd4,0x3f,
	0x3c,0xdd,0xbd,0xe3,
);

static void check_secret(tls_hkdf_t *hkdf, tls_hkdf_label_t label, chunk_t data, chunk_t exp_secret)
{
	chunk_t secret;

	ck_assert(hkdf->generate_secret(hkdf, label, data, &secret));
	ck_assert_chunk_eq(exp_secret, secret);

	chunk_free(&secret);
}

static void check_secret_key_iv(tls_hkdf_t *hkdf, tls_hkdf_label_t label,
								chunk_t data, bool is_server, chunk_t exp_secret,
								int key_length, int iv_length, chunk_t exp_key,
								chunk_t exp_iv)
{
	chunk_t key, iv;

	check_secret(hkdf, label, data, exp_secret);

	ck_assert(hkdf->derive_key(hkdf, is_server, key_length, &key));
	ck_assert_chunk_eq(exp_key, key);

	ck_assert(hkdf->derive_iv(hkdf, is_server, iv_length, &iv));
	ck_assert_chunk_eq(exp_iv, iv);

	chunk_free(&key);
	chunk_free(&iv);
}

static void check_finished(tls_hkdf_t *hkdf, bool is_server, chunk_t exp_finished)
{
	chunk_t finished;

	ck_assert(hkdf->derive_finished(hkdf, is_server, &finished));
	ck_assert_chunk_eq(exp_finished, finished);

	chunk_free(&finished);
}

static void check_resumption(tls_hkdf_t *hkdf, chunk_t data, chunk_t exp_resume)
{
	chunk_t nonce, resume;

	nonce = chunk_from_chars(0x00,0x00);
	ck_assert(hkdf->resume(hkdf, data, nonce, &resume));
	ck_assert_chunk_eq(exp_resume, resume);

	chunk_free(&resume);
}

START_TEST(test_ulfheim_handshake)
{
	chunk_t exp_client_handshake_traffic_secret = chunk_from_chars(
		0xff,0x0e,0x5b,0x96,0x52,0x91,0xc6,0x08,0xc1,0xe8,0xcd,0x26,0x7e,0xef,0xc0,0xaf,
		0xcc,0x5e,0x98,0xa2,0x78,0x63,0x73,0xf0,0xdb,0x47,0xb0,0x47,0x86,0xd7,0x2a,0xea,
	);

	chunk_t exp_client_handshake_key = chunk_from_chars(
		0x71,0x54,0xf3,0x14,0xe6,0xbe,0x7d,0xc0,0x08,0xdf,0x2c,0x83,0x2b,0xaa,0x1d,0x39,
	);

	chunk_t exp_client_handshake_iv = chunk_from_chars(
		0x71,0xab,0xc2,0xca,0xe4,0xc6,0x99,0xd4,0x7c,0x60,0x02,0x68,
	);

	chunk_t exp_server_handshake_traffic_secret = chunk_from_chars(
		0xa2,0x06,0x72,0x65,0xe7,0xf0,0x65,0x2a,0x92,0x3d,0x5d,0x72,0xab,0x04,0x67,0xc4,
		0x61,0x32,0xee,0xb9,0x68,0xb6,0xa3,0x2d,0x31,0x1c,0x80,0x58,0x68,0x54,0x88,0x14,
	);

	chunk_t exp_server_handshake_key = chunk_from_chars(
		0x84,0x47,0x80,0xa7,0xac,0xad,0x9f,0x98,0x0f,0xa2,0x5c,0x11,0x4e,0x43,0x40,0x2a,
	);

	chunk_t exp_server_handshake_iv = chunk_from_chars(
		0x4c,0x04,0x2d,0xdc,0x12,0x0a,0x38,0xd1,0x41,0x7f,0xc8,0x15,
	);

	chunk_t exp_client_finished_key = chunk_from_chars(
		0x7c,0x60,0xf8,0xd6,0x34,0x6f,0x4a,0x96,0x91,0xd2,0xae,0x64,0x5a,0x78,0x85,0xe0,
		0x10,0x4a,0xdf,0xf9,0x8e,0xba,0x98,0x1c,0xa2,0xf9,0x9e,0xf6,0x2b,0xdd,0x8f,0xaa,
	);

	chunk_t exp_server_finished_key = chunk_from_chars(
		0xea,0x84,0xab,0xd2,0xad,0xa0,0xb5,0xc6,0x4c,0x08,0x07,0xa3,0x26,0xb6,0xfd,0x94,
		0xa9,0x59,0x7e,0x39,0xca,0x62,0x10,0x60,0x7c,0x0d,0x3c,0x8c,0x76,0x68,0x65,0x71,
	);

	tls_hkdf_t *hkdf = tls_hkdf_create(HASH_SHA256, chunk_empty);
	ck_assert(hkdf);

	hkdf->set_shared_secret(hkdf, ulfheim_ecdhe);

	/* Generate client handshake traffic secret */
	check_secret_key_iv(hkdf, TLS_HKDF_C_HS_TRAFFIC, ulfheim_client_server_hello,
						FALSE, exp_client_handshake_traffic_secret, 16, 12,
						exp_client_handshake_key, exp_client_handshake_iv);

	check_finished(hkdf, FALSE, exp_client_finished_key);

	/* Generate server handshake traffic secret */
	check_secret_key_iv(hkdf, TLS_HKDF_S_HS_TRAFFIC, ulfheim_client_server_hello,
						TRUE, exp_server_handshake_traffic_secret, 16, 12,
						exp_server_handshake_key, exp_server_handshake_iv);

	check_finished(hkdf, TRUE, exp_server_finished_key);

	hkdf->destroy(hkdf);
}
END_TEST

START_TEST(test_ulfheim_traffic)
{
	chunk_t exp_client_application_traffic_secret = chunk_from_chars(
		0xb8,0x82,0x22,0x31,0xc1,0xd6,0x76,0xec,0xca,0x1c,0x11,0xff,0xf6,0x59,0x42,0x80,
		0x31,0x4d,0x03,0xa4,0xe9,0x1c,0xf1,0xaf,0x7f,0xe7,0x3f,0x8f,0x7b,0xe2,0xc1,0x1b,
	);

	chunk_t exp_client_application_key = chunk_from_chars(
		0x49,0x13,0x4b,0x95,0x32,0x8f,0x27,0x9f,0x01,0x83,0x86,0x05,0x89,0xac,0x67,0x07,
	);

	chunk_t exp_client_application_iv = chunk_from_chars(
		0xbc,0x4d,0xd5,0xf7,0xb9,0x8a,0xcf,0xf8,0x54,0x66,0x26,0x1d,
	);

	chunk_t exp_server_application_traffic_secret = chunk_from_chars(
		0x3f,0xc3,0x5e,0xa7,0x06,0x93,0x06,0x9a,0x27,0x79,0x56,0xaf,0xa2,0x3b,0x8f,0x45,
		0x43,0xce,0x68,0xac,0x59,0x5f,0x2a,0xac,0xe0,0x5c,0xd7,0xa1,0xc9,0x20,0x23,0xd5,
	);

	chunk_t exp_server_application_key = chunk_from_chars(
		0x0b,0x6d,0x22,0xc8,0xff,0x68,0x09,0x7e,0xa8,0x71,0xc6,0x72,0x07,0x37,0x73,0xbf,
	);

	chunk_t exp_server_application_iv = chunk_from_chars(
		0x1b,0x13,0xdd,0x9f,0x8d,0x8f,0x17,0x09,0x1d,0x34,0xb3,0x49,
	);

	chunk_t hs_data;

	tls_hkdf_t *hkdf = tls_hkdf_create(HASH_SHA256, chunk_empty);
	ck_assert(hkdf);

	hkdf->set_shared_secret(hkdf, ulfheim_ecdhe);

	/* Generate client application traffic secret */
	hs_data = chunk_cata("cc", ulfheim_client_server_hello, ulfheim_server_data);
	check_secret_key_iv(hkdf, TLS_HKDF_C_AP_TRAFFIC, hs_data, FALSE,
						exp_client_application_traffic_secret, 16, 12,
						exp_client_application_key, exp_client_application_iv);

	/* Generate server application traffic secret */
	check_secret_key_iv(hkdf, TLS_HKDF_S_AP_TRAFFIC, hs_data, TRUE,
						exp_server_application_traffic_secret, 16, 12,
						exp_server_application_key, exp_server_application_iv);

	hkdf->destroy(hkdf);
}
END_TEST

START_TEST(test_rfc8448_simple_1_rtt_handshake)
{
	chunk_t client_hello = chunk_from_chars(
		0x01,0x00,0x00,0xc0,0x03,0x03,0xcb,0x34,0xec,0xb1,0xe7,0x81,0x63,0xba,0x1c,0x38,
		0xc6,0xda,0xcb,0x19,0x6a,0x6d,0xff,0xa2,0x1a,0x8d,0x99,0x12,0xec,0x18,0xa2,0xef,
		0x62,0x83,0x02,0x4d,0xec,0xe7,0x00,0x00,0x06,0x13,0x01,0x13,0x03,0x13,0x02,0x01,
		0x00,0x00,0x91,0x00,0x00,0x00,0x0b,0x00,0x09,0x00,0x00,0x06,0x73,0x65,0x72,0x76,
		0x65,0x72,0xff,0x01,0x00,0x01,0x00,0x00,0x0a,0x00,0x14,0x00,0x12,0x00,0x1d,0x00,
		0x17,0x00,0x18,0x00,0x19,0x01,0x00,0x01,0x01,0x01,0x02,0x01,0x03,0x01,0x04,0x00,
		0x23,0x00,0x00,0x00,0x33,0x00,0x26,0x00,0x24,0x00,0x1d,0x00,0x20,0x99,0x38,0x1d,
		0xe5,0x60,0xe4,0xbd,0x43,0xd2,0x3d,0x8e,0x43,0x5a,0x7d,0xba,0xfe,0xb3,0xc0,0x6e,
		0x51,0xc1,0x3c,0xae,0x4d,0x54,0x13,0x69,0x1e,0x52,0x9a,0xaf,0x2c,0x00,0x2b,0x00,
		0x03,0x02,0x03,0x04,0x00,0x0d,0x00,0x20,0x00,0x1e,0x04,0x03,0x05,0x03,0x06,0x03,
		0x02,0x03,0x08,0x04,0x08,0x05,0x08,0x06,0x04,0x01,0x05,0x01,0x06,0x01,0x02,0x01,
		0x04,0x02,0x05,0x02,0x06,0x02,0x02,0x02,0x00,0x2d,0x00,0x02,0x01,0x01,0x00,0x1c,
		0x00,0x02,0x40,0x01,
	);

	chunk_t server_hello = chunk_from_chars(
		0x02,0x00,0x00,0x56,0x03,0x03,0xa6,0xaf,0x06,0xa4,0x12,0x18,0x60,0xdc,0x5e,0x6e,
		0x60,0x24,0x9c,0xd3,0x4c,0x95,0x93,0x0c,0x8a,0xc5,0xcb,0x14,0x34,0xda,0xc1,0x55,
		0x77,0x2e,0xd3,0xe2,0x69,0x28,0x00,0x13,0x01,0x00,0x00,0x2e,0x00,0x33,0x00,0x24,
		0x00,0x1d,0x00,0x20,0xc9,0x82,0x88,0x76,0x11,0x20,0x95,0xfe,0x66,0x76,0x2b,0xdb,
		0xf7,0xc6,0x72,0xe1,0x56,0xd6,0xcc,0x25,0x3b,0x83,0x3d,0xf1,0xdd,0x69,0xb1,0xb0,
		0x4e,0x75,0x1f,0x0f,0x00,0x2b,0x00,0x02,0x03,0x04,
	);

	chunk_t server_data = chunk_from_chars(
		/* Server Encrypted Extension */
		0x08,0x00,0x00,0x24,0x00,0x22,0x00,0x0a,0x00,0x14,0x00,0x12,0x00,0x1d,0x00,0x17,
		0x00,0x18,0x00,0x19,0x01,0x00,0x01,0x01,0x01,0x02,0x01,0x03,0x01,0x04,0x00,0x1c,
		0x00,0x02,0x40,0x01,0x00,0x00,0x00,0x00,
		/* Server Certificate */
		0x0b,0x00,0x01,0xb9,0x00,0x00,0x01,0xb5,0x00,0x01,0xb0,0x30,0x82,0x01,0xac,0x30,
		0x82,0x01,0x15,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x02,0x30,0x0d,0x06,0x09,0x2a,
		0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x30,0x0e,0x31,0x0c,0x30,0x0a,
		0x06,0x03,0x55,0x04,0x03,0x13,0x03,0x72,0x73,0x61,0x30,0x1e,0x17,0x0d,0x31,0x36,
		0x30,0x37,0x33,0x30,0x30,0x31,0x32,0x33,0x35,0x39,0x5a,0x17,0x0d,0x32,0x36,0x30,
		0x37,0x33,0x30,0x30,0x31,0x32,0x33,0x35,0x39,0x5a,0x30,0x0e,0x31,0x0c,0x30,0x0a,
		0x06,0x03,0x55,0x04,0x03,0x13,0x03,0x72,0x73,0x61,0x30,0x81,0x9f,0x30,0x0d,0x06,
		0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
		0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb4,0xbb,0x49,0x8f,0x82,0x79,0x30,0x3d,0x98,
		0x08,0x36,0x39,0x9b,0x36,0xc6,0x98,0x8c,0x0c,0x68,0xde,0x55,0xe1,0xbd,0xb8,0x26,
		0xd3,0x90,0x1a,0x24,0x61,0xea,0xfd,0x2d,0xe4,0x9a,0x91,0xd0,0x15,0xab,0xbc,0x9a,
		0x95,0x13,0x7a,0xce,0x6c,0x1a,0xf1,0x9e,0xaa,0x6a,0xf9,0x8c,0x7c,0xed,0x43,0x12,
		0x09,0x98,0xe1,0x87,0xa8,0x0e,0xe0,0xcc,0xb0,0x52,0x4b,0x1b,0x01,0x8c,0x3e,0x0b,
		0x63,0x26,0x4d,0x44,0x9a,0x6d,0x38,0xe2,0x2a,0x5f,0xda,0x43,0x08,0x46,0x74,0x80,
		0x30,0x53,0x0e,0xf0,0x46,0x1c,0x8c,0xa9,0xd9,0xef,0xbf,0xae,0x8e,0xa6,0xd1,0xd0,
		0x3e,0x2b,0xd1,0x93,0xef,0xf0,0xab,0x9a,0x80,0x02,0xc4,0x74,0x28,0xa6,0xd3,0x5a,
		0x8d,0x88,0xd7,0x9f,0x7f,0x1e,0x3f,0x02,0x03,0x01,0x00,0x01,0xa3,0x1a,0x30,0x18,
		0x30,0x09,0x06,0x03,0x55,0x1d,0x13,0x04,0x02,0x30,0x00,0x30,0x0b,0x06,0x03,0x55,
		0x1d,0x0f,0x04,0x04,0x03,0x02,0x05,0xa0,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
		0xf7,0x0d,0x01,0x01,0x0b,0x05,0x00,0x03,0x81,0x81,0x00,0x85,0xaa,0xd2,0xa0,0xe5,
		0xb9,0x27,0x6b,0x90,0x8c,0x65,0xf7,0x3a,0x72,0x67,0x17,0x06,0x18,0xa5,0x4c,0x5f,
		0x8a,0x7b,0x33,0x7d,0x2d,0xf7,0xa5,0x94,0x36,0x54,0x17,0xf2,0xea,0xe8,0xf8,0xa5,
		0x8c,0x8f,0x81,0x72,0xf9,0x31,0x9c,0xf3,0x6b,0x7f,0xd6,0xc5,0x5b,0x80,0xf2,0x1a,
		0x03,0x01,0x51,0x56,0x72,0x60,0x96,0xfd,0x33,0x5e,0x5e,0x67,0xf2,0xdb,0xf1,0x02,
		0x70,0x2e,0x60,0x8c,0xca,0xe6,0xbe,0xc1,0xfc,0x63,0xa4,0x2a,0x99,0xbe,0x5c,0x3e,
		0xb7,0x10,0x7c,0x3c,0x54,0xe9,0xb9,0xeb,0x2b,0xd5,0x20,0x3b,0x1c,0x3b,0x84,0xe0,
		0xa8,0xb2,0xf7,0x59,0x40,0x9b,0xa3,0xea,0xc9,0xd9,0x1d,0x40,0x2d,0xcc,0x0c,0xc8,
		0xf8,0x96,0x12,0x29,0xac,0x91,0x87,0xb4,0x2b,0x4d,0xe1,0x00,0x00,
		/* Server Certificate Verify */
		0x0f,0x00,0x00,0x84,0x08,0x04,0x00,0x80,0x5a,0x74,0x7c,0x5d,0x88,0xfa,0x9b,0xd2,
		0xe5,0x5a,0xb0,0x85,0xa6,0x10,0x15,0xb7,0x21,0x1f,0x82,0x4c,0xd4,0x84,0x14,0x5a,
		0xb3,0xff,0x52,0xf1,0xfd,0xa8,0x47,0x7b,0x0b,0x7a,0xbc,0x90,0xdb,0x78,0xe2,0xd3,
		0x3a,0x5c,0x14,0x1a,0x07,0x86,0x53,0xfa,0x6b,0xef,0x78,0x0c,0x5e,0xa2,0x48,0xee,
		0xaa,0xa7,0x85,0xc4,0xf3,0x94,0xca,0xb6,0xd3,0x0b,0xbe,0x8d,0x48,0x59,0xee,0x51,
		0x1f,0x60,0x29,0x57,0xb1,0x54,0x11,0xac,0x02,0x76,0x71,0x45,0x9e,0x46,0x44,0x5c,
		0x9e,0xa5,0x8c,0x18,0x1e,0x81,0x8e,0x95,0xb8,0xc3,0xfb,0x0b,0xf3,0x27,0x84,0x09,
		0xd3,0xbe,0x15,0x2a,0x3d,0xa5,0x04,0x3e,0x06,0x3d,0xda,0x65,0xcd,0xf5,0xae,0xa2,
		0x0d,0x53,0xdf,0xac,0xd4,0x2f,0x74,0xf3,
		/* Server Handshake Finish */
		0x14,0x00,0x00,0x20,0x9b,0x9b,0x14,0x1d,0x90,0x63,0x37,0xfb,0xd2,0xcb,0xdc,0xe7,
		0x1d,0xf4,0xde,0xda,0x4a,0xb4,0x2c,0x30,0x95,0x72,0xcb,0x7f,0xff,0xee,0x54,0x54,
		0xb7,0x8f,0x07,0x18,
	);

	chunk_t client_finished = chunk_from_chars(
		0x14,0x00,0x00,0x20,0xa8,0xec,0x43,0x6d,0x67,0x76,0x34,0xae,0x52,0x5a,0xc1,0xfc,
		0xeb,0xe1,0x1a,0x03,0x9e,0xc1,0x76,0x94,0xfa,0xc6,0xe9,0x85,0x27,0xb6,0x42,0xf2,
		0xed,0xd5,0xce,0x61,
	);

	chunk_t ecdhe = chunk_from_chars(
		0x8b,0xd4,0x05,0x4f,0xb5,0x5b,0x9d,0x63,0xfd,0xfb,0xac,0xf9,0xf0,0x4b,0x9f,0x0d,
		0x35,0xe6,0xd6,0x3f,0x53,0x75,0x63,0xef,0xd4,0x62,0x72,0x90,0x0f,0x89,0x49,0x2d,
	);

	chunk_t exp_client_handshake_traffic_secret = chunk_from_chars(
		0xb3,0xed,0xdb,0x12,0x6e,0x06,0x7f,0x35,0xa7,0x80,0xb3,0xab,0xf4,0x5e,0x2d,0x8f,
		0x3b,0x1a,0x95,0x07,0x38,0xf5,0x2e,0x96,0x00,0x74,0x6a,0x0e,0x27,0xa5,0x5a,0x21,
	);

	chunk_t exp_client_handshake_key = chunk_from_chars(
		0xdb,0xfa,0xa6,0x93,0xd1,0x76,0x2c,0x5b,0x66,0x6a,0xf5,0xd9,0x50,0x25,0x8d,0x01,
	);

	chunk_t exp_client_handshake_iv = chunk_from_chars(
		0x5b,0xd3,0xc7,0x1b,0x83,0x6e,0x0b,0x76,0xbb,0x73,0x26,0x5f,
	);

	chunk_t exp_server_handshake_traffic_secret = chunk_from_chars(
		0xb6,0x7b,0x7d,0x69,0x0c,0xc1,0x6c,0x4e,0x75,0xe5,0x42,0x13,0xcb,0x2d,0x37,0xb4,
		0xe9,0xc9,0x12,0xbc,0xde,0xd9,0x10,0x5d,0x42,0xbe,0xfd,0x59,0xd3,0x91,0xad,0x38,
	);

	chunk_t exp_server_handshake_key = chunk_from_chars(
		0x3f,0xce,0x51,0x60,0x09,0xc2,0x17,0x27,0xd0,0xf2,0xe4,0xe8,0x6e,0xe4,0x03,0xbc,
	);

	chunk_t exp_server_handshake_iv = chunk_from_chars(
		0x5d,0x31,0x3e,0xb2,0x67,0x12,0x76,0xee,0x13,0x00,0x0b,0x30,
	);

	chunk_t exp_client_finished_key = chunk_from_chars(
		0xb8,0x0a,0xd0,0x10,0x15,0xfb,0x2f,0x0b,0xd6,0x5f,0xf7,0xd4,0xda,0x5d,0x6b,0xf8,
		0x3f,0x84,0x82,0x1d,0x1f,0x87,0xfd,0xc7,0xd3,0xc7,0x5b,0x5a,0x7b,0x42,0xd9,0xc4,
	);

	chunk_t exp_server_finished_key = chunk_from_chars(
		0x00,0x8d,0x3b,0x66,0xf8,0x16,0xea,0x55,0x9f,0x96,0xb5,0x37,0xe8,0x85,0xc3,0x1f,
		0xc0,0x68,0xbf,0x49,0x2c,0x65,0x2f,0x01,0xf2,0x88,0xa1,0xd8,0xcd,0xc1,0x9f,0xc8,
	);

	chunk_t exp_client_application_traffic_secret = chunk_from_chars(
		0x9e,0x40,0x64,0x6c,0xe7,0x9a,0x7f,0x9d,0xc0,0x5a,0xf8,0x88,0x9b,0xce,0x65,0x52,
		0x87,0x5a,0xfa,0x0b,0x06,0xdf,0x00,0x87,0xf7,0x92,0xeb,0xb7,0xc1,0x75,0x04,0xa5,
	);

	chunk_t exp_client_application_key = chunk_from_chars(
		0x17,0x42,0x2d,0xda,0x59,0x6e,0xd5,0xd9,0xac,0xd8,0x90,0xe3,0xc6,0x3f,0x50,0x51,
	);

	chunk_t exp_client_application_iv = chunk_from_chars(
		0x5b,0x78,0x92,0x3d,0xee,0x08,0x57,0x90,0x33,0xe5,0x23,0xd9,
	);

	chunk_t exp_server_application_traffic_secret = chunk_from_chars(
		0xa1,0x1a,0xf9,0xf0,0x55,0x31,0xf8,0x56,0xad,0x47,0x11,0x6b,0x45,0xa9,0x50,0x32,
		0x82,0x04,0xb4,0xf4,0x4b,0xfb,0x6b,0x3a,0x4b,0x4f,0x1f,0x3f,0xcb,0x63,0x16,0x43,
	);

	chunk_t exp_server_application_key = chunk_from_chars(
		0x9f,0x02,0x28,0x3b,0x6c,0x9c,0x07,0xef,0xc2,0x6b,0xb9,0xf2,0xac,0x92,0xe3,0x56,
	);

	chunk_t exp_server_application_iv = chunk_from_chars(
		0xcf,0x78,0x2b,0x88,0xdd,0x83,0x54,0x9a,0xad,0xf1,0xe9,0x84,
	);

	chunk_t exp_generated_resumption_secret = chunk_from_chars(
		0x4e,0xcd,0x0e,0xb6,0xec,0x3b,0x4d,0x87,0xf5,0xd6,0x02,0x8f,0x92,0x2c,0xa4,0xc5,
		0x85,0x1a,0x27,0x7f,0xd4,0x13,0x11,0xc9,0xe6,0x2d,0x2c,0x94,0x92,0xe1,0xc4,0xf3,
	);

	chunk_t hs_data;

	tls_hkdf_t *hkdf = tls_hkdf_create(HASH_SHA256, chunk_empty);
	ck_assert(hkdf);

	hkdf->set_shared_secret(hkdf, ecdhe);

	/* Generate client handshake traffic secret */
	hs_data = chunk_cata("cc", client_hello, server_hello);
	check_secret_key_iv(hkdf, TLS_HKDF_C_HS_TRAFFIC, hs_data, FALSE,
						exp_client_handshake_traffic_secret, 16, 12,
						exp_client_handshake_key,
						exp_client_handshake_iv);
	check_finished(hkdf, FALSE, exp_client_finished_key);

	/* Generate server handshake traffic secret */
	check_secret_key_iv(hkdf, TLS_HKDF_S_HS_TRAFFIC, hs_data, TRUE,
						exp_server_handshake_traffic_secret, 16, 12,
						exp_server_handshake_key,
						exp_server_handshake_iv);
	check_finished(hkdf, TRUE, exp_server_finished_key);

	/* Generate client application traffic secret */
	hs_data = chunk_cata("cc", hs_data, server_data);
	check_secret_key_iv(hkdf, TLS_HKDF_C_AP_TRAFFIC, hs_data, FALSE,
						exp_client_application_traffic_secret, 16, 12,
						exp_client_application_key,
						exp_client_application_iv);

	/* Generate server application traffic secret */
	check_secret_key_iv(hkdf, TLS_HKDF_S_AP_TRAFFIC, hs_data, TRUE,
						exp_server_application_traffic_secret, 16, 12,
						exp_server_application_key,
						exp_server_application_iv);

	/* Generating resumption master secret */
	hs_data = chunk_cata("cc", hs_data, client_finished);
	check_resumption(hkdf, hs_data, exp_generated_resumption_secret);

	hkdf->destroy(hkdf);
}
END_TEST

START_TEST(test_rfc8448_resumed_0_rtt_handshake)
{
	chunk_t client_hello = chunk_from_chars(
		0x01,0x00,0x01,0xfc,0x03,0x03,0x1b,0xc3,0xce,0xb6,0xbb,0xe3,0x9c,0xff,0x93,0x83,
		0x55,0xb5,0xa5,0x0a,0xdb,0x6d,0xb2,0x1b,0x7a,0x6a,0xf6,0x49,0xd7,0xb4,0xbc,0x41,
		0x9d,0x78,0x76,0x48,0x7d,0x95,0x00,0x00,0x06,0x13,0x01,0x13,0x03,0x13,0x02,0x01,
		0x00,0x01,0xcd,0x00,0x00,0x00,0x0b,0x00,0x09,0x00,0x00,0x06,0x73,0x65,0x72,0x76,
		0x65,0x72,0xff,0x01,0x00,0x01,0x00,0x00,0x0a,0x00,0x14,0x00,0x12,0x00,0x1d,0x00,
		0x17,0x00,0x18,0x00,0x19,0x01,0x00,0x01,0x01,0x01,0x02,0x01,0x03,0x01,0x04,0x00,
		0x33,0x00,0x26,0x00,0x24,0x00,0x1d,0x00,0x20,0xe4,0xff,0xb6,0x8a,0xc0,0x5f,0x8d,
		0x96,0xc9,0x9d,0xa2,0x66,0x98,0x34,0x6c,0x6b,0xe1,0x64,0x82,0xba,0xdd,0xda,0xfe,
		0x05,0x1a,0x66,0xb4,0xf1,0x8d,0x66,0x8f,0x0b,0x00,0x2a,0x00,0x00,0x00,0x2b,0x00,
		0x03,0x02,0x03,0x04,0x00,0x0d,0x00,0x20,0x00,0x1e,0x04,0x03,0x05,0x03,0x06,0x03,
		0x02,0x03,0x08,0x04,0x08,0x05,0x08,0x06,0x04,0x01,0x05,0x01,0x06,0x01,0x02,0x01,
		0x04,0x02,0x05,0x02,0x06,0x02,0x02,0x02,0x00,0x2d,0x00,0x02,0x01,0x01,0x00,0x1c,
		0x00,0x02,0x40,0x01,0x00,0x15,0x00,0x57,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x29,0x00,0xdd,0x00,0xb8,0x00,0xb2,0x2c,0x03,0x5d,0x82,0x93,0x59,0xee,0x5f,0xf7,
		0xaf,0x4e,0xc9,0x00,0x00,0x00,0x00,0x26,0x2a,0x64,0x94,0xdc,0x48,0x6d,0x2c,0x8a,
		0x34,0xcb,0x33,0xfa,0x90,0xbf,0x1b,0x00,0x70,0xad,0x3c,0x49,0x88,0x83,0xc9,0x36,
		0x7c,0x09,0xa2,0xbe,0x78,0x5a,0xbc,0x55,0xcd,0x22,0x60,0x97,0xa3,0xa9,0x82,0x11,
		0x72,0x83,0xf8,0x2a,0x03,0xa1,0x43,0xef,0xd3,0xff,0x5d,0xd3,0x6d,0x64,0xe8,0x61,
		0xbe,0x7f,0xd6,0x1d,0x28,0x27,0xdb,0x27,0x9c,0xce,0x14,0x50,0x77,0xd4,0x54,0xa3,
		0x66,0x4d,0x4e,0x6d,0xa4,0xd2,0x9e,0xe0,0x37,0x25,0xa6,0xa4,0xda,0xfc,0xd0,0xfc,
		0x67,0xd2,0xae,0xa7,0x05,0x29,0x51,0x3e,0x3d,0xa2,0x67,0x7f,0xa5,0x90,0x6c,0x5b,
		0x3f,0x7d,0x8f,0x92,0xf2,0x28,0xbd,0xa4,0x0d,0xda,0x72,0x14,0x70,0xf9,0xfb,0xf2,
		0x97,0xb5,0xae,0xa6,0x17,0x64,0x6f,0xac,0x5c,0x03,0x27,0x2e,0x97,0x07,0x27,0xc6,
		0x21,0xa7,0x91,0x41,0xef,0x5f,0x7d,0xe6,0x50,0x5e,0x5b,0xfb,0xc3,0x88,0xe9,0x33,
		0x43,0x69,0x40,0x93,0x93,0x4a,0xe4,0xd3,0x57,0xfa,0xd6,0xaa,0xcb,
	);

	chunk_t client_hello_hash = chunk_from_chars(
		0x63,0x22,0x4b,0x2e,0x45,0x73,0xf2,0xd3,0x45,0x4c,0xa8,0x4b,0x9d,0x00,0x9a,0x04,
		0xf6,0xbe,0x9e,0x05,0x71,0x1a,0x83,0x96,0x47,0x3a,0xef,0xa0,0x1e,0x92,0x4a,0x14,
	);

	chunk_t server_hello = chunk_from_chars(
		0x02,0x00,0x00,0x5c,0x03,0x03,0x3c,0xcf,0xd2,0xde,0xc8,0x90,0x22,0x27,0x63,0x47,
		0x2a,0xe8,0x13,0x67,0x77,0xc9,0xd7,0x35,0x87,0x77,0xbb,0x66,0xe9,0x1e,0xa5,0x12,
		0x24,0x95,0xf5,0x59,0xea,0x2d,0x00,0x13,0x01,0x00,0x00,0x34,0x00,0x29,0x00,0x02,
		0x00,0x00,0x00,0x33,0x00,0x24,0x00,0x1d,0x00,0x20,0x12,0x17,0x61,0xee,0x42,0xc3,
		0x33,0xe1,0xb9,0xe7,0x7b,0x60,0xdd,0x57,0xc2,0x05,0x3c,0xd9,0x45,0x12,0xab,0x47,
		0xf1,0x15,0xe8,0x6e,0xff,0x50,0x94,0x2c,0xea,0x31,0x00,0x2b,0x00,0x02,0x03,0x04,
	);

	chunk_t encrypted_extension = chunk_from_chars(
		0x08,0x00,0x00,0x28,0x00,0x26,0x00,0x0a,0x00,0x14,0x00,0x12,0x00,0x1d,0x00,0x17,
		0x00,0x18,0x00,0x19,0x01,0x00,0x01,0x01,0x01,0x02,0x01,0x03,0x01,0x04,0x00,0x1c,
		0x00,0x02,0x40,0x01,0x00,0x00,0x00,0x00,0x00,0x2a,0x00,0x00,
	);

	chunk_t server_finished = chunk_from_chars(
		0x14,0x00,0x00,0x20,0x48,0xd3,0xe0,0xe1,0xb3,0xd9,0x07,0xc6,0xac,0xff,0x14,0x5e,
		0x16,0x09,0x03,0x88,0xc7,0x7b,0x05,0xc0,0x50,0xb6,0x34,0xab,0x1a,0x88,0xbb,0xd0,
		0xdd,0x1a,0x34,0xb2,
	);

	chunk_t end_of_early_data = chunk_from_chars(
		0x05,0x00,0x00,0x00,
	);

	chunk_t client_finished = chunk_from_chars(
		0x14,0x00,0x00,0x20,0x72,0x30,0xa9,0xc9,0x52,0xc2,0x5c,0xd6,0x13,0x8f,0xc5,0xe6,
		0x62,0x83,0x08,0xc4,0x1c,0x53,0x35,0xdd,0x81,0xb9,0xf9,0x6b,0xce,0xa5,0x0f,0xd3,
		0x2b,0xda,0x41,0x6d,
	);

	chunk_t psk = chunk_from_chars(
		0x4e,0xcd,0x0e,0xb6,0xec,0x3b,0x4d,0x87,0xf5,0xd6,0x02,0x8f,0x92,0x2c,0xa4,0xc5,
		0x85,0x1a,0x27,0x7f,0xd4,0x13,0x11,0xc9,0xe6,0x2d,0x2c,0x94,0x92,0xe1,0xc4,0xf3,
	);

	chunk_t ecdhe = chunk_from_chars(
		0xf4,0x41,0x94,0x75,0x6f,0xf9,0xec,0x9d,0x25,0x18,0x06,0x35,0xd6,0x6e,0xa6,0x82,
		0x4c,0x6a,0xb3,0xbf,0x17,0x99,0x77,0xbe,0x37,0xf7,0x23,0x57,0x0e,0x7c,0xcb,0x2e,
	);

	chunk_t exp_psk_binder = chunk_from_chars(
		0x3a,0xdd,0x4f,0xb2,0xd8,0xfd,0xf8,0x22,0xa0,0xca,0x3c,0xf7,0x67,0x8e,0xf5,0xe8,
		0x8d,0xae,0x99,0x01,0x41,0xc5,0x92,0x4d,0x57,0xbb,0x6f,0xa3,0x1b,0x9e,0x5f,0x9d,
	);

	chunk_t exp_early_exporter_master_secret = chunk_from_chars(
		0xb2,0x02,0x68,0x66,0x61,0x09,0x37,0xd7,0x42,0x3e,0x5b,0xe9,0x08,0x62,0xcc,0xf2,
		0x4c,0x0e,0x60,0x91,0x18,0x6d,0x34,0xf8,0x12,0x08,0x9f,0xf5,0xbe,0x2e,0xf7,0xdf,
	);

	chunk_t exp_client_handshake_traffic_secret = chunk_from_chars(
		0x2f,0xaa,0xc0,0x8f,0x85,0x1d,0x35,0xfe,0xa3,0x60,0x4f,0xcb,0x4d,0xe8,0x2d,0xc6,
		0x2c,0x9b,0x16,0x4a,0x70,0x97,0x4d,0x04,0x62,0xe2,0x7f,0x1a,0xb2,0x78,0x70,0x0f,
	);

	chunk_t exp_client_handshake_key = chunk_from_chars(
		0xb1,0x53,0x08,0x06,0xf4,0xad,0xfe,0xac,0x83,0xf1,0x41,0x30,0x32,0xbb,0xfa,0x82,
	);

	chunk_t exp_client_handshake_iv = chunk_from_chars(
		0xeb,0x50,0xc1,0x6b,0xe7,0x65,0x4a,0xbf,0x99,0xdd,0x06,0xd9,
	);

	chunk_t exp_server_handshake_traffic_secret = chunk_from_chars(
		0xfe,0x92,0x7a,0xe2,0x71,0x31,0x2e,0x8b,0xf0,0x27,0x5b,0x58,0x1c,0x54,0xee,0xf0,
		0x20,0x45,0x0d,0xc4,0xec,0xff,0xaa,0x05,0xa1,0xa3,0x5d,0x27,0x51,0x8e,0x78,0x03,
	);

	chunk_t exp_server_handshake_key = chunk_from_chars(
		0x27,0xc6,0xbd,0xc0,0xa3,0xdc,0xea,0x39,0xa4,0x73,0x26,0xd7,0x9b,0xc9,0xe4,0xee,
	);

	chunk_t exp_server_handshake_iv = chunk_from_chars(
		0x95,0x69,0xec,0xdd,0x4d,0x05,0x36,0x70,0x5e,0x9e,0xf7,0x25,
	);

	chunk_t exp_server_finished = chunk_from_chars(
		0x4b,0xb7,0x4c,0xae,0x7a,0x5d,0xc8,0x91,0x46,0x04,0xc0,0xbf,0xbe,0x2f,0x0c,0x06,
		0x23,0x96,0x88,0x39,0x22,0xbe,0xc8,0xa1,0x5e,0x2a,0x9b,0x53,0x2a,0x5d,0x39,0x2c,

	);

	chunk_t exp_client_finished = chunk_from_chars(
		0x5a,0xce,0x39,0x4c,0x26,0x98,0x0d,0x58,0x12,0x43,0xf6,0x27,0xd1,0x15,0x0a,0xe2,
		0x7e,0x37,0xfa,0x52,0x36,0x4e,0x0a,0x7f,0x20,0xac,0x68,0x6d,0x09,0xcd,0x0e,0x8e,
	);

	chunk_t exp_client_application_traffic_secret = chunk_from_chars(
		0x2a,0xbb,0xf2,0xb8,0xe3,0x81,0xd2,0x3d,0xbe,0xbe,0x1d,0xd2,0xa7,0xd1,0x6a,0x8b,
		0xf4,0x84,0xcb,0x49,0x50,0xd2,0x3f,0xb7,0xfb,0x7f,0xa8,0x54,0x70,0x62,0xd9,0xa1,
	);

	chunk_t exp_client_application_key = chunk_from_chars(
		0x3c,0xf1,0x22,0xf3,0x01,0xc6,0x35,0x8c,0xa7,0x98,0x95,0x53,0x25,0x0e,0xfd,0x72,
	);

	chunk_t exp_client_application_iv = chunk_from_chars(
		0xab,0x1a,0xec,0x26,0xaa,0x78,0xb8,0xfc,0x11,0x76,0xb9,0xac,
	);

	chunk_t exp_server_application_traffic_secret = chunk_from_chars(
		0xcc,0x21,0xf1,0xbf,0x8f,0xeb,0x7d,0xd5,0xfa,0x50,0x5b,0xd9,0xc4,0xb4,0x68,0xa9,
		0x98,0x4d,0x55,0x4a,0x99,0x3d,0xc4,0x9e,0x6d,0x28,0x55,0x98,0xfb,0x67,0x26,0x91,
	);

	chunk_t exp_server_application_key = chunk_from_chars(
		0xe8,0x57,0xc6,0x90,0xa3,0x4c,0x5a,0x91,0x29,0xd8,0x33,0x61,0x96,0x84,0xf9,0x5e
	);

	chunk_t exp_server_application_iv = chunk_from_chars(
		0x06,0x85,0xd6,0xb5,0x61,0xaa,0xb9,0xef,0x10,0x13,0xfa,0xf9,
	);

	chunk_t exp_exporter_master_secret = chunk_from_chars(
		0x3f,0xd9,0x3d,0x4f,0xfd,0xdc,0x98,0xe6,0x4b,0x14,0xdd,0x10,0x7a,0xed,0xf8,0xee,
		0x4a,0xdd,0x23,0xf4,0x51,0x0f,0x58,0xa4,0x59,0x2d,0x0b,0x20,0x1b,0xee,0x56,0xb4,
	);

	chunk_t exp_resumption_master_secret = chunk_from_chars(
		0x5e,0x95,0xbd,0xf1,0xf8,0x90,0x05,0xea,0x2e,0x9a,0xa0,0xba,0x85,0xe7,0x28,0xe3,
		0xc1,0x9c,0x5f,0xe0,0xc6,0x99,0xe3,0xf5,0xbe,0xe5,0x9f,0xae,0xbd,0x0b,0x54,0x06,
	);

	chunk_t hs_data, psk_binder;

	tls_hkdf_t *hkdf = tls_hkdf_create(HASH_SHA256, psk);
	ck_assert(hkdf);

	ck_assert(hkdf->binder(hkdf, client_hello_hash, &psk_binder));
	ck_assert_chunk_eq(exp_psk_binder, psk_binder);

	/* PSK binder is wrapped first with 0x20 and then with 0x00,0x21 length bytes*/
	hs_data = chunk_cata("ccc", client_hello, chunk_from_chars(0x00,0x21,0x20),
						 psk_binder);
	check_secret(hkdf, TLS_HKDF_E_EXP_MASTER, hs_data, exp_early_exporter_master_secret);

	hkdf->set_shared_secret(hkdf, ecdhe);

	/* Generate client handshake traffic secret */
	hs_data = chunk_cata("cc", hs_data, server_hello);
	check_secret_key_iv(hkdf, TLS_HKDF_C_HS_TRAFFIC, hs_data, FALSE,
						exp_client_handshake_traffic_secret, 16, 12,
						exp_client_handshake_key,
						exp_client_handshake_iv);

	/* Generate sever handshake traffic secret */
	check_secret_key_iv(hkdf, TLS_HKDF_S_HS_TRAFFIC, hs_data, TRUE,
						exp_server_handshake_traffic_secret, 16, 12,
						exp_server_handshake_key,
						exp_server_handshake_iv);

	check_finished(hkdf, TRUE, exp_server_finished);
	check_finished(hkdf, FALSE, exp_client_finished);

	/* Generate client application traffic secret */
	hs_data = chunk_cata("ccc", hs_data, encrypted_extension, server_finished);
	check_secret_key_iv(hkdf, TLS_HKDF_C_AP_TRAFFIC, hs_data, FALSE,
						exp_client_application_traffic_secret, 16, 12,
						exp_client_application_key,
						exp_client_application_iv);

	/* Generate server application traffic secret */
	check_secret_key_iv(hkdf, TLS_HKDF_S_AP_TRAFFIC, hs_data, TRUE,
						exp_server_application_traffic_secret, 16, 12,
						exp_server_application_key,
						exp_server_application_iv);

	check_secret(hkdf, TLS_HKDF_EXP_MASTER, hs_data, exp_exporter_master_secret);

	hs_data = chunk_cata("ccc", hs_data, end_of_early_data, client_finished);
	check_secret(hkdf, TLS_HKDF_RES_MASTER, hs_data, exp_resumption_master_secret);

	hkdf->destroy(hkdf);
	chunk_free(&psk_binder);
}
END_TEST

Suite *hkdf_suite_create()
{
	Suite *s;
	TCase *tc;

	s = suite_create("HKDF TLS 1.3");

	tc = tcase_create("Ulfheim Keys");
	tcase_add_test(tc, test_ulfheim_handshake);
	tcase_add_test(tc, test_ulfheim_traffic);
	suite_add_tcase(s, tc);

	tc = tcase_create("RFC 8448");
	tcase_add_test(tc, test_rfc8448_simple_1_rtt_handshake);
	tcase_add_test(tc, test_rfc8448_resumed_0_rtt_handshake);
	suite_add_tcase(s, tc);

	return s;
}
